--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Quests
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

-- Quest class definition
LQuest =
{
	__tostring = function(t) return "{Quest:" .. t.name .. "}" end,
	_ByIndex = {},
	_ByName = {},
	_Variables = {},
	_StringFiles = {},
	
	hintweeks = 8,				-- TODO: Set default hintweeks
}

-------------------------------------------------------------------------------
-- Functions for data description

function DefineQuest(t) return LQuest:new(t) end

-------------------------------------------------------------------------------
-- "static" functions to access global lists

function LQuest:AllQuests()
	return bsgArrayIterator(self._ByIndex)
end

function LQuest:ByName(name)
	return self._ByName[name]
end

function LQuest:ByIndex(i)
	return self._ByIndex[i]
end

function LQuest:Variable(var)
	self._Variables[var] = self._Variables[var] or 0
	return self._Variables[var]
end

function LQuest:IncrementVariable(var)
	self._Variables[var] = self._Variables[var] or 0
	self._Variables[var] = self._Variables[var] + 1
end

-------------------------------------------------------------------------------
-- "constructor"

function LQuest:new(t)
	-- Standard object creation...
	t = t or {} setmetatable(t, self) self.__index = self
	
	-- Check parameters
	if not t.name then
		bsgDevWarning("Quest defined with no name")
	elseif self._ByName[t.name] then
		bsgDevWarning("Quest " .. t.name .. " already defined")
	elseif not t.starter then
		bsgDevWarning("Quest " .. t.name .. " has no starter")
	else
		-- Keep global tables
		self._ByName[t.name] = t
		table.insert(self._ByIndex, t)
		
		-- Interpret "forceOffer" and "priority" into forceTime
		if not t.forceTime then
			if t.forceOffer then t.forceTime = 0
			elseif t.priority then t.forceTime = 12
			end
		end
		
		if type(t.starter) == "string" then t.starter = { t.starter } end
		if t.ender then t.explicitEnder = true
		else t.ender = t.starter
		end
		if type(t.ender) == "string" then t.ender = { t.ender } end
		
		return t
	end
	
	return t
end

-------------------------------------------------------------------------------
-- force Offer

function LQuest:CheckForceOffer(time)
	return (self.forceTime and time >= self.forceTime)
end

-------------------------------------------------------------------------------
-- Quest Timer

function LQuest:ResetQuestTimer()
	-- A "REAL" quest resets the force quest timer
	-- Real quests are things that require players to take a specific action,
	-- either accomplishing a goal or going to find another specific character
	if self.goals or self.explicitEnder then
		gSim.questWeek = gSim.weeks
	end
end

-------------------------------------------------------------------------------
-- Reset

function LQuest:ClearQuests()
	LQuest._ByIndex = {}
	LQuest._ByName = {}
	LQuest._Variables = {}
	LQuest._StringFiles = {}
end

function LQuest:ResetAll()
	for _,q in ipairs(self._ByIndex) do
		q.complete = nil
		q.eligibleTime = 0
	end
	
	for v,_ in pairs(self._Variables) do
		self._Variables[v] = 0
	end
end

-------------------------------------------------------------------------------
-- Data post-processing

function LQuest:PostProcessAll()
	local totalCount = 0
	local repeatableCount = 0

	for name,q in pairs(self._ByIndex) do
		totalCount = totalCount + 1
		if q.repeatable then repeatableCount = repeatableCount + 1 end
	
		-- Cross-reference characters
		for i,cn in ipairs(q.starter) do
			if type(cn) == "string" then
				local c = LCharacter:ByName(cn)
				if c then
					q.starter[i] = c
					table.insert(c.quests, q)
				else
					bsgDevWarning("Quest "..tostring(q)..": Starter "..cn.." undefined")
				end
			end
		end

		for i,cn in ipairs(q.ender) do
			if type(cn) == "string" then
				local c = LCharacter:ByName(cn)
				if c then
					q.ender[i] = c
					table.insert(c.endQuests, q)
				else
					bsgDevWarning("Quest "..tostring(q)..": Ender "..cn.." undefined")
				end
			elseif cn.endQuests then
				table.insert(cn.endQuests, q)
			end
		end
		
		-- Cross-reference follow-up and dependent quests
		if q.followUp then
			local next = LQuest:ByName(q.followUp)
			if not next then
				bsgDevWarning("Quest "..tostring(q).." followUp: no quest '"..q.followUp.."' found")
			else
				next.requirements = next.requirements or {}
				table.insert(next.requirements, RequireQuest(q.name))
			end
		end
	end
	
DebugOut("QUESTS: " .. tostring(totalCount) .. " -- REPEATABLE: " ..tostring(repeatableCount))
	
end

-------------------------------------------------------------------------------
-- Load and Save

function LQuest:BuildSaveTable()
	local t = {}
	for _,q in ipairs(self._ByIndex) do
		local data = nil
		if q.complete then data = true
		elseif q.eligibleTime > gSim.weeks then data = q.eligibleTime
		end
		if data then t[q.name] = data end
	end
	
	t._Variables = self._Variables
	return t
end

function LQuest:LoadSaveTable(t)
	self._Variables = t._Variables
	t._Variables = nil
	
	for name,data in pairs(t) do
		local q = LQuest:ByName(name)
		if q then
			if type(data) == "number" then q.eligibleTime = data
			else q.complete = true
			end
		end
	end
end

-------------------------------------------------------------------------------
-- Quest strings

function QuestStringTable(f)
	table.insert(LQuest._StringFiles, f)
	bsgLoadStringFile(f)
end

function LQuest:ReloadStringTable()
	for _,f in ipairs(self._StringFiles) do
		bsgLoadStringFile(f)
	end
end

local function ProcessQuestText(s)
	local var = GetString(s)
	if var == s then var = nil end
	return var
end

function LQuest:NameText()
	local s = GetString(self.name)
	if s == self.name or s == "#####" then s = nil end
	return s
end

function LQuest:SummaryText()
	local s = self.summary or self.name.."_summary"
	return ProcessQuestText(s)
end

function LQuest:IntroText()
	local s = self.intro or self.name.."_intro"
	return ProcessQuestText(s)
end

function LQuest:IncompleteText()
	local s = self.incomplete or self.name.."_incomplete"
	return ProcessQuestText(s)
end

function LQuest:HintText()
	local s = self.hint
	return ProcessQuestText(s)
end

function LQuest:CompleteText()
	local s = self.complete_text or self.name.."_complete"
	return ProcessQuestText(s)
end

-------------------------------------------------------------------------------
-- Eligibility and Completion

function LQuest:IsEligible()
	local eligible = (not self.eligibleTime) or (self.eligibleTime <= gSim.weeks)
	eligible = eligible and (not self.complete)
	if eligible then
		if self.requirements then
			for _,f in ipairs(self.requirements) do
				eligible = f()
				if not eligible then break end
			end
		end
	end
	return eligible
end

function LQuest:EvaluateGoals()
	local complete = true
	if self.goals then
		for _,f in ipairs(self.goals) do
			complete = f(nil, self)
			if not complete then break end
		end
	end
	return complete
end

-------------------------------------------------------------------------------
-- Rollover

local incompleteColoring = "<font color='ee2222'>"

function LQuest:Rollover()
	if self.goals then
		local s = "#"..GetString("q_goals")
		local r =
		{
			x=0,y=0, color=PopupColor, inset=2, alpha=.8,
			AppendStyle { font = popupFont, flags=kHAlignLeft+kVAlignTop },
			TightText { x=0,y=0, label=s },
		}
		local dy = bsgFontHeight(popupFont)
		local y = dy
	
		local visible = false
		for _,f in ipairs(self.goals) do
			local label,show = f(true)
			if not label then label = "" end
			if show then
				visible = true
				if not f() then label = incompleteColoring..label end
				table.insert(r, TightText { x=0,y=y, label="#"..label })
				y = y + dy
			end
		end
		
		if visible then return MakeRollover(r) end
	end
	
	return nil
end

-------------------------------------------------------------------------------
-- Activation and Completion

function LQuest:Offer(character)
	character = character or self.starter[1]
	local text = self:IntroText()
	if text then
		local yn = "yes"
		if self.forceAccept then
			-- FIRSTPEEK: QuestAccept, time-stamp, weeks, quest-name
			if fpWrite then fpWrite { "QuestAccept", gSim.weeks, self.name } end
		
			DisplayDialog { "ui/chartalk.lua", char=character, body="#"..text,
				ok=self.accept or "ok" }
		else
			yn = DisplayDialog { "ui/chartalk.lua", char=character, body="#"..text,
				yes=self.accept or "yes", no=self.reject or "no" }
		end
		if yn == "yes" then
			-- FIRSTPEEK: QuestAccept, time-stamp, weeks, quest-name
			if fpWrite then fpWrite { "QuestAccept", gSim.weeks, self.name } end

			self:Activate()
		else
			-- FIRSTPEEK: QuestReject, time-stamp, weeks, quest-name
			if fpWrite then fpWrite { "QuestReject", gSim.weeks, self.name } end
		
			self:Reject(character)
		end
	else
		-- Quest with no intro text is forceAccept by default
		self:Activate(character)
	end

	if DevUpdateCurrent then DevUpdateCurrent() end
	
	
	if gSim.quest == self and self:EvaluateGoals() then
		-- If this character is also an ender, do an immediate quest completion (forceComplete)
		local ok = false
		for _,c in ipairs(self.ender) do
			if c == character then
				ok = true
				break
			end
		end
		if ok then
			self:Complete(character)
			if DevUpdateCurrent then DevUpdateCurrent() end
		end
	end
end

function LQuest:Reject(character)
	local n = self:NameText()
	if n then gSim:QueueMessage(GetString("q_rejected", n)) end
	
	-- By default, don't offer for another week
	-- Repeatable quests: offer immediate repeat as desired, may be overridden by specific "DelayQuest"
	if (not self.repeatable) and (self.eligibleTime <= gSim.weeks) then
		self.eligibleTime = gSim.weeks + 1
	end
	
	if self.denial then
		questCharacter = character or self.starter[1]
		for _,f in ipairs(self.denial) do f() end
		questCharacter = nil
	end
	
	gSim:FlushMessages()
end

function LQuest:Activate(character)
	local n = self:NameText()
	if n then gSim:QueueMessage(GetString("q_accepted", n)) end

	gSim.quest = self
	self:ResetQuestTimer()
	gSim.hintWeek = gSim.weeks + self.hintweeks
	if self.gifts then
		questCharacter = character or self.starter[1]
		for _,f in ipairs(self.gifts) do f() end
		questCharacter = nil
	end
	
	local s = self:SummaryText() or ""
	SetQuestHint(s)
	if DevUpdateCurrent then DevUpdateCurrent() end

	-- Auto-detect hint strings for this quest
	if not self.hint then
		local key = self.name .. "_hint"
		local hint = GetString(key)
		if hint ~= key then self.hint = key end
	end
	
	gSim:FlushMessages()
end

function LQuest:Complete(character)
	-- First Peek: QuestComplete, <time stamp>, current game time, quest name, game time required,
	-- TODO: Real time required ?
	-- FIRSTPEEK: QuestComplete, <stamp>, weeks, quest-name, quest-weeks
	if fpWrite then fpWrite { "QuestComplete", gSim.weeks, self.name, gSim.weeks - gSim.questWeek } end

	self:ResetQuestTimer()
	
	character = character or self.ender[1]
	local text = self:CompleteText()
	if text then
		DisplayDialog { "ui/chartalk.lua", char=character, body="#"..text, ok=self.finish or "ok" }
	end

	if self.repeatable then
		-- repeatable quests may be repeated as often as desired...
--		self.eligibleTime = gSim.weeks + 1
		self.eligibleTime = gSim.weeks
	else
		self.complete = true
	end
	if gSim.quest == self then gSim.quest = nil end
	
	local n = self:NameText()
	if n then gSim:QueueMessage(GetString("q_completed", n)) end
	if self.rewards then
		questCharacter = character
		for _,f in ipairs(self.rewards) do f() end
		questCharacter = nil
	end
	
	-- Offer followUp quest if there is one
	if self.followUp and not gSim.quest then
		local follow = LQuest:ByName(self.followUp)
		if follow then follow:Offer(character) end
	end
	
	if not gSim.quest then SetQuestHint("") end
	if DevUpdateCurrent then DevUpdateCurrent() end
	
	gSim:FlushMessages()
end

function LQuest:Incomplete(character)
	character = character or self.ender[1]
	
	-- Should we use a hint, or the regular Incomplete text?
	local text = nil

--	if self.hint and gSim.hintWeek <= gSim.weeks then
--		text = self:HintText()
--		gSim.hintWeek = gSim.hintWeek + 12
--	else
		text = self:IncompleteText()
--	end
	
	if text then
		DisplayDialog { "ui/chartalk.lua", char=character, body="#"..text, ok="ok" }
	end
end

function LQuest:CheckCompletion(character, allowIncomplete)
	-- Is this character an ender for this quest?
	local ok = false
	for _,c in ipairs(self.ender) do
		if c == character then
			ok = true
			break
		end
	end
	
	if ok then
		if self:EvaluateGoals() then self:Complete(character)
		elseif allowIncomplete then self:Incomplete(character)
		else ok = false
		end
	end
	
	return ok
end
